<?php
header('Content-Type: application/json');

require_once __DIR__ . '/../core/config.php';
require_once __DIR__ . '/../core/Database.php';
require_once __DIR__ . '/../models/InvoiceModel.php';

$response = [
    'success' => false,
    'message' => '',
    'errors' => []
];

try {
    // Validate request method
    if ($_SERVER["REQUEST_METHOD"] !== "POST") {
        throw new Exception("Invalid request method");
    }

    // Validate required fields
    $requiredFields = ['invoice_id', 'amount', 'payment_method', 'timestamp'];
    foreach ($requiredFields as $field) {
        if (empty($_POST[$field])) {
            $response['errors'][$field] = "This field is required";
        }
    }

    // If any validation errors, return them
    if (!empty($response['errors'])) {
        $response['message'] = "Validation failed";
        echo json_encode($response);
        exit;
    }

    // Sanitize and validate input
    $invoice_id = (int)$_POST['invoice_id'];
    $amount = (float)$_POST['amount'];
    $payment_method = htmlspecialchars($_POST['payment_method']);
    $timestamp = $_POST['timestamp'];
    $client_id = isset($_POST['client_id']) ? (int)$_POST['client_id'] : null;
    $notes = isset($_POST['payment_notes']) ? htmlspecialchars($_POST['payment_notes']) : '';

    // Additional validation
    if ($invoice_id <= 0) {
        throw new Exception("Invalid invoice ID");
    }

    if ($amount <= 0) {
        throw new Exception("Payment amount must be greater than 0");
    }

    if (!strtotime($timestamp)) {
        throw new Exception("Invalid payment date");
    }

    // Process payment
    $invoiceModel = new InvoiceModel();
    
    // Verify invoice exists and belongs to client (if client_id provided)
    if ($client_id) {
        $invoice = $invoiceModel->getInvoiceDetails($invoice_id);
        if (!$invoice || $invoice['client_id'] != $client_id) {
            throw new Exception("Invoice not found for this client");
        }
    }

    // Record payment
    if ($invoiceModel->addPayment($invoice_id, $amount, $payment_method, $timestamp, $notes)) {
        $response['success'] = true;
        $response['message'] = "Payment recorded successfully";
        
        // Get updated invoice details for response
        $updatedInvoice = $invoiceModel->getInvoiceDetails($invoice_id);
        $response['invoice'] = [
            'paid_amount' => $updatedInvoice['paid_amount'],
            'due_amount' => $updatedInvoice['due_amount'],
            'status' => $updatedInvoice['status']
        ];
    } else {
        throw new Exception("Failed to record payment");
    }

} catch (Exception $e) {
    $response['message'] = $e->getMessage();
    http_response_code(400); // Bad request
}

echo json_encode($response);