<?php


class BillModel {

    private $conn;

    // Constructor - initialize database connection
    public function __construct() {
        $database = new Database();
        $this->conn = $database->getConnection();
    }

    // Create a new Bill and return the bill ID
    public function createBill($client, $date, $status) {
        $query = "INSERT INTO `bill` (`client`, `date`, `status`) VALUES (?, ?, ?)";
        $stmt = $this->conn->prepare($query);
        $stmt->bind_param("sss", $client, $date, $status);

        if ($stmt->execute()) {
            return $this->conn->insert_id; // Return the bill ID
        }
        return false;
    }

    // Retrieve all Bills
    public function getAllBills() {
        $query = "SELECT `id`, `client`, `date`, `status` FROM `bill`";
        $result = $this->conn->query($query);
        $bills = [];
        while ($row = $result->fetch_assoc()) {
            $bills[] = $row;
        }
        return $bills;
    }

    // Retrieve a specific Bill by ID along with its Items
    public function getBillById($billId) {
        $query = "SELECT * FROM `bill` WHERE `id` = ?";
        $stmt = $this->conn->prepare($query);
        $stmt->bind_param("i", $billId);
        $stmt->execute();
        $bill = $stmt->get_result()->fetch_assoc();

        if ($bill) {
            $bill['items'] = $this->getBillItemsByBillId($billId);
        }
        return $bill;
    }

    // Update a Bill by ID
    public function updateBill($id, $client, $date, $status) {
        $query = "UPDATE `bill` SET `client` = ?, `date` = ?, `status` = ? WHERE `id` = ?";
        $stmt = $this->conn->prepare($query);
        $stmt->bind_param("sssi", $client, $date, $status, $id);
        return $stmt->execute();
    }

    // Delete a Bill (cascade delete items)
    public function deleteBill($id) {
        $query = "DELETE FROM `bill` WHERE `id` = ?";
        $stmt = $this->conn->prepare($query);
        $stmt->bind_param("i", $id);
        return $stmt->execute();
    }

    // Add multiple items to a Bill
    public function addBillItems($billId, $items) {
        $query = "INSERT INTO `billItem` (`billId`, `ProductName`, `Quantity`, `Description`, `UnitPrice`) VALUES (?, ?, ?, ?, ?)";
        $stmt = $this->conn->prepare($query);

        foreach ($items as $item) {
            $stmt->bind_param("isisi", $billId, $item['ProductName'], $item['Quantity'], $item['Description'], $item['UnitPrice']);
            $stmt->execute();
        }
        return true;
    }

    // Retrieve all Items for a Bill
    public function getBillItemsByBillId($billId) {
        $query = "SELECT * FROM `billItem` WHERE `billId` = ?";
        $stmt = $this->conn->prepare($query);
        $stmt->bind_param("i", $billId);
        $stmt->execute();
        $result = $stmt->get_result();
        $items = [];
        while ($row = $result->fetch_assoc()) {
            $items[] = $row;
        }
        return $items;
    }

    // Update a Bill Item
    public function updateBillItem($id, $productName, $quantity, $description, $unitPrice) {
        $query = "UPDATE `billItem` SET `ProductName` = ?, `Quantity` = ?, `Description` = ?, `UnitPrice` = ? WHERE `id` = ?";
        $stmt = $this->conn->prepare($query);
        $stmt->bind_param("sisii", $productName, $quantity, $description, $unitPrice, $id);
        return $stmt->execute();
    }

    // Delete a Bill Item
    public function deleteBillItem($id) {
        $query = "DELETE FROM `billItem` WHERE `id` = ?";
        $stmt = $this->conn->prepare($query);
        $stmt->bind_param("i", $id);
        return $stmt->execute();
    }
    
        // Get total bill amount for this month
    public function getTotalBillThisMonth() {
        $query = "SELECT SUM(bi.Quantity * bi.UnitPrice) AS total FROM billItem bi
                  JOIN bill b ON bi.billId = b.id
                  WHERE MONTH(b.date) = MONTH(CURRENT_DATE()) AND YEAR(b.date) = YEAR(CURRENT_DATE())";
        $result = $this->conn->query($query);
        $row = $result->fetch_assoc();
        return $row['total'] ?? 0;
    }

    // Get total due amount for this month (Unpaid & Pending)
    public function getTotalDueThisMonth() {
        $query = "SELECT SUM(bi.Quantity * bi.UnitPrice) AS total_due FROM billItem bi
                  JOIN bill b ON bi.billId = b.id
                  WHERE (b.status = 'Unpaid' OR b.status = 'Pending') 
                  AND MONTH(b.date) = MONTH(CURRENT_DATE()) AND YEAR(b.date) = YEAR(CURRENT_DATE())";
        $result = $this->conn->query($query);
        $row = $result->fetch_assoc();
        return $row['total_due'] ?? 0;
    }

    // Get total collected amount for this month (Paid bills)
    public function getTotalCollectedThisMonth() {
        $query = "SELECT SUM(bi.Quantity * bi.UnitPrice) AS total_collected FROM billItem bi
                  JOIN bill b ON bi.billId = b.id
                  WHERE b.status = 'Paid' 
                  AND MONTH(b.date) = MONTH(CURRENT_DATE()) AND YEAR(b.date) = YEAR(CURRENT_DATE())";
        $result = $this->conn->query($query);
        $row = $result->fetch_assoc();
        return $row['total_collected'] ?? 0;
    }
public function getMonthlyBillData()
{
    $query = "SELECT MONTH(b.date) AS month, 
                     SUM(bi.Quantity * bi.UnitPrice) AS total_bill, 
                     SUM(CASE WHEN b.status = 'Paid' THEN (bi.Quantity * bi.UnitPrice) ELSE 0 END) AS total_paid,
                     SUM(CASE WHEN b.status IN ('Unpaid', 'Pending') THEN (bi.Quantity * bi.UnitPrice) ELSE 0 END) AS total_pending
              FROM bill b
              JOIN billItem bi ON b.id = bi.billId
              WHERE YEAR(b.date) = YEAR(CURDATE())
              GROUP BY MONTH(b.date)
              ORDER BY MONTH(b.date)";

    $result = $this->conn->query($query);
    $data = [];

    while ($row = mysqli_fetch_assoc($result)) {
        $data[] = $row;
    }

    return $data;
}


}
?>
