<?php
class ClientModel {
    private $conn;

    public function __construct() {
        $database = new Database();
        $this->conn = $database->getConnection();
    }

    // Create a new client
    public function createClient($institution, $clientName, $phoneNumber, $address, $username, $serviceCondition) {
        $query = "INSERT INTO client (Institution, ClientName, PhoneNumber, address, username, serviceCondition) VALUES (?, ?, ?, ?, ?, ?)";
        $stmt = $this->conn->prepare($query);
        $stmt->bind_param("ssssss", $institution, $clientName, $phoneNumber, $address, $username, $serviceCondition);
        return $stmt->execute();
    }

// Get all clients
public function getAllClients() {
    $query = "SELECT id, Institution, ClientName, PhoneNumber, address, username, serviceCondition FROM client";
    $stmt = $this->conn->prepare($query);
    $stmt->execute();
    
    $stmt->bind_result($id, $institution, $clientName, $phoneNumber, $address, $username, $serviceCondition);
    
    $clients = [];
    while ($stmt->fetch()) {
        $clients[] = [
            'id' => $id,
            'Institution' => $institution,
            'ClientName' => $clientName,
            'PhoneNumber' => $phoneNumber,
            'address' => $address,
            'username' => $username,
            'serviceCondition' => $serviceCondition
        ];
    }
    
    $stmt->close();
    return $clients;
}

// Get client by ID
public function getClientById($id) {
    $query = "SELECT id, Institution, ClientName, PhoneNumber, address, username, serviceCondition FROM client WHERE id = ?";
    $stmt = $this->conn->prepare($query);
    $stmt->bind_param("i", $id);
    $stmt->execute();
    
    $stmt->bind_result($id, $institution, $clientName, $phoneNumber, $address, $username, $serviceCondition);
    $stmt->fetch();
    
    $client = [
        'id' => $id,
        'Institution' => $institution,
        'ClientName' => $clientName,
        'PhoneNumber' => $phoneNumber,
        'address' => $address,
        'username' => $username,
        'serviceCondition' => $serviceCondition
    ];
    
    $stmt->close();
    return $client;
}


    // Update a client
    public function updateClient($id, $institution, $clientName, $phoneNumber, $address, $username, $serviceCondition) {
        $query = "UPDATE client SET Institution = ?, ClientName = ?, PhoneNumber = ?, address = ?, username = ?, serviceCondition = ? WHERE id = ?";
        $stmt = $this->conn->prepare($query);
        $stmt->bind_param("ssssssi", $institution, $clientName, $phoneNumber, $address, $username, $serviceCondition, $id);
        return $stmt->execute();
    }

    // Delete a client
    public function deleteClient($id) {
        $query = "DELETE FROM client WHERE id = ?";
        $stmt = $this->conn->prepare($query);
        $stmt->bind_param("i", $id);
        return $stmt->execute();
    }
    
    
    // Function to update the service condition of a client
    public function updateServiceCondition($clientId, $newCondition) {
        // Sanitize the input to prevent SQL injection
        $clientId = mysqli_real_escape_string($this->conn, $clientId);
        $newCondition = mysqli_real_escape_string($this->conn, $newCondition);

        // SQL query to update the service condition in the clients table
        $sql = "UPDATE client SET serviceCondition = ? WHERE id = ?";

        // Prepare the query
        if ($stmt = mysqli_prepare($this->conn, $sql)) {
            // Bind the parameters
            mysqli_stmt_bind_param($stmt, "si", $newCondition, $clientId);

            // Execute the query
            if (mysqli_stmt_execute($stmt)) {
                return true;
            } else {
                return false;
            }

            // Close the statement
            mysqli_stmt_close($stmt);
        } else {
            return false;
        }
    }    
    
    
public function getInvoiceSummary() {
    $query = "
        SELECT 
            c.id,
            c.Institution, 
            c.ClientName, 
            c.PhoneNumber, 
            COUNT(i.id) AS TotalInvoices, 
            SUM(i.total_amount) AS TotalAmount, 
            SUM(i.paid_amount) AS TotalPaid, 
            SUM(i.due_amount) AS TotalDue
        FROM client c
        LEFT JOIN i_invoices i ON c.id = i.client_id
        GROUP BY c.id
    ";

    $stmt = $this->conn->prepare($query);
    $stmt->execute();
    $stmt->bind_result($id, $institution, $clientName, $phoneNumber, $totalInvoices, $totalAmount, $totalPaid, $totalDue);

    $invoices = [];
    while ($stmt->fetch()) {
        $invoices[] = [
            'id' => $id,
            'Institution' => $institution,
            'ClientName' => $clientName,
            'PhoneNumber' => $phoneNumber,
            'TotalInvoices' => $totalInvoices,
            'TotalAmount' => $totalAmount,
            'TotalPaid' => $totalPaid,
            'TotalDue' => $totalDue
        ];
    }

    $stmt->close();
    return $invoices;
}
    
    
    
}