<?php
class InvoiceModel {
    private $conn;
    private $table_invoices = "i_invoices";
    private $table_items = "i_invoice_items";
    private $table_payments = "i_payments";
    private $table_status = "i_invoice_status_history";

    public function __construct() {
        $database = new Database();
        $this->conn = $database->getConnection();
    }

    // Create a new invoice
public function createInvoice($client_id, $total_amount) {
    // Insert the invoice with initial details
    $query = "INSERT INTO $this->table_invoices (client_id, total_amount, paid_amount, due_amount, status) 
              VALUES (?, ?, 0, ?, 'Pending')";
    $stmt = $this->conn->prepare($query);
    $stmt->bind_param("idd", $client_id, $total_amount, $total_amount);
    
    if ($stmt->execute()) {
        // Get the last inserted ID
        $invoice_id = $this->conn->insert_id;
        
        // Generate invoice number in the format INV-{id}
        $invoice_number = 'INV-' . $invoice_id;
        
        // Update the invoice with the generated invoice number
        $update_query = "UPDATE $this->table_invoices SET invoice_number = ? WHERE id = ?";
        $update_stmt = $this->conn->prepare($update_query);
        $update_stmt->bind_param("si", $invoice_number, $invoice_id);
        
        if ($update_stmt->execute()) {
            // Log status change (if needed)
            $this->logStatusChange($invoice_id, 'Pending');
            return $invoice_id;
        }
    }
    return false;
}

    // Retrieve all Bills
    public function getAllBills() {
        $query = "SELECT `id`, `client_id`, `invoice_number`, `total_amount`, `paid_amount`, `due_amount`, `status`, `date` FROM `i_invoices` WHERE 1";
        $result = $this->conn->query($query);
        $bills = [];
        while ($row = $result->fetch_assoc()) {
            $bills[] = $row;
        }
        return $bills;
    }
    
// Get total invoice amount for this month
public function getTotalInvoiceThisMonth() {
    $query = "SELECT SUM(bi.Quantity * bi.unit_price) AS total
              FROM i_invoice_items bi
              JOIN i_invoices inv ON bi.invoice_id = inv.id
              WHERE MONTH(inv.date) = MONTH(CURRENT_DATE()) 
              AND YEAR(inv.date) = YEAR(CURRENT_DATE())";
    
    // Check if the query is successful
    $result = $this->conn->query($query);
    
    if (!$result) {
        // Log the error or display a message
        die("Query failed: " . $this->conn->error);
    }
    
    $row = $result->fetch_assoc();
    return $row['total'] ?? 0;
}

    
    
    // Add items to an invoice
public function addInvoiceItem($invoice_id, $product_name, $quantity, $unit_price, $description) {
    $total_price = $quantity * $unit_price; // Calculate total price
    $query = "INSERT INTO $this->table_items (invoice_id, product_name, quantity, unit_price, description, total_price) 
              VALUES (?, ?, ?, ?, ?, ?)";
    $stmt = $this->conn->prepare($query);
    $stmt->bind_param("isidss", $invoice_id, $product_name, $quantity, $unit_price, $description, $total_price);
    return $stmt->execute();
}


// Record a payment for an invoice
public function addPayment($invoice_id, $amount, $payment_method, $payment_date) {
    $query = "INSERT INTO $this->table_payments (invoice_id, amount, payment_method, payment_date) 
              VALUES (?, ?, ?, ?)";
    $stmt = $this->conn->prepare($query);
    $stmt->bind_param("idss", $invoice_id, $amount, $payment_method, $payment_date);

    if ($stmt->execute()) {
        return $this->updateInvoicePayment($invoice_id, $amount);
    }
    return false;
}


    // Update invoice payment and status
private function updateInvoicePayment($invoice_id, $amount) {
    $query = "UPDATE $this->table_invoices 
              SET paid_amount = paid_amount + ?, 
                  due_amount = due_amount - ?, 
                  status = CASE 
                      WHEN (paid_amount + ? - paid_amount) >= total_amount THEN 'Paid'
                      ELSE 'Partially Paid' 
                  END
              WHERE id = ?";
    $stmt = $this->conn->prepare($query);
    $stmt->bind_param("dddi", $amount, $amount, $amount, $invoice_id);

    if ($stmt->execute()) {
        return $this->logStatusChange($invoice_id, $this->getInvoiceStatus($invoice_id));
    }
    return false;
}

    // Log invoice status change
    private function logStatusChange($invoice_id, $status) {
        $query = "INSERT INTO $this->table_status (invoice_id, status, changed_at) 
                  VALUES (?, ?, NOW())";
        $stmt = $this->conn->prepare($query);
        $stmt->bind_param("is", $invoice_id, $status);
        return $stmt->execute();
    }

    // Get invoice status
    private function getInvoiceStatus($invoice_id) {
        $query = "SELECT status FROM $this->table_invoices WHERE id = ?";
        $stmt = $this->conn->prepare($query);
        $stmt->bind_param("i", $invoice_id);
        $stmt->execute();
        $stmt->bind_result($status);
        $stmt->fetch();
        return $status;
    }

    // Get invoice details (including items & payments)
    public function getInvoiceDetails($invoice_id) {
        $query = "SELECT * FROM $this->table_invoices WHERE id = ?";
        $stmt = $this->conn->prepare($query);
        $stmt->bind_param("i", $invoice_id);
        $stmt->execute();
        $result = $stmt->get_result();
        $invoice = $result->fetch_assoc();

        if ($invoice) {
            $invoice['items'] = $this->getInvoiceItems($invoice_id);
            $invoice['payments'] = $this->getInvoicePayments($invoice_id);
        }
        return $invoice;
    }

    // Get invoice items
    private function getInvoiceItems($invoice_id) {
        $query = "SELECT * FROM $this->table_items WHERE invoice_id = ?";
        $stmt = $this->conn->prepare($query);
        $stmt->bind_param("i", $invoice_id);
        $stmt->execute();
        $result = $stmt->get_result();
        return $result->fetch_all(MYSQLI_ASSOC);
    }

    // Get payments for an invoice
    private function getInvoicePayments($invoice_id) {
        $query = "SELECT * FROM $this->table_payments WHERE invoice_id = ?";
        $stmt = $this->conn->prepare($query);
        $stmt->bind_param("i", $invoice_id);
        $stmt->execute();
        $result = $stmt->get_result();
        return $result->fetch_all(MYSQLI_ASSOC);
    }

    // Get payment history
    public function getPaymentHistory($client_id) {
        $query = "SELECT p.*, i.invoice_number 
                  FROM $this->table_payments p 
                  JOIN $this->table_invoices i ON p.invoice_id = i.id 
                  WHERE i.client_id = ? 
                  ORDER BY p.payment_date DESC";
        $stmt = $this->conn->prepare($query);
        $stmt->bind_param("i", $client_id);
        $stmt->execute();
        $result = $stmt->get_result();
        return $result->fetch_all(MYSQLI_ASSOC);
    }

    // Generate report (total paid and due for each client)
    public function generateReport() {
        $query = "SELECT i.client_id, c.name, SUM(i.total_amount) AS total_billed, 
                         SUM(i.paid_amount) AS total_paid, SUM(i.due_amount) AS total_due 
                  FROM $this->table_invoices i 
                  JOIN clients c ON i.client_id = c.id 
                  GROUP BY i.client_id";
        $stmt = $this->conn->prepare($query);
        $stmt->execute();
        $result = $stmt->get_result();
        return $result->fetch_all(MYSQLI_ASSOC);
    }

    // Get all invoices with status
    public function getAllInvoices() {
        $query = "SELECT * FROM $this->table_invoices ORDER BY date DESC";
        $stmt = $this->conn->prepare($query);
        $stmt->execute();
        $result = $stmt->get_result();
        return $result->fetch_all(MYSQLI_ASSOC);
    }
// Get total due amount for this month
public function getTotalDueThisMonth() {
    $query = "SELECT SUM(due_amount) AS total_due
              FROM i_invoices
              WHERE MONTH(date) = MONTH(CURRENT_DATE()) 
              AND YEAR(date) = YEAR(CURRENT_DATE())";
    
    $result = $this->conn->query($query);
    
    if (!$result) {
        die("Query failed: " . $this->conn->error);
    }
    
    $row = $result->fetch_assoc();
    return $row['total_due'] ?? 0;
}
    
    // Get total collected (paid) amount for this month
public function getTotalCollectedThisMonth() {
    $query = "SELECT SUM(amount) AS total_collected
              FROM i_payments
              WHERE MONTH(payment_date) = MONTH(CURRENT_DATE()) 
              AND YEAR(payment_date) = YEAR(CURRENT_DATE())";
    
    $result = $this->conn->query($query);
    
    if (!$result) {
        die("Query failed: " . $this->conn->error);
    }
    
    $row = $result->fetch_assoc();
    return $row['total_collected'] ?? 0;
}

public function getMonthlyBillData() {
    $query = "SELECT YEAR(date) AS year, MONTH(date) AS month, 
                     SUM(paid_amount) AS total_paid, 
                     SUM(due_amount) AS total_pending
              FROM i_invoices
              GROUP BY YEAR(date), MONTH(date)
              ORDER BY YEAR(date) DESC, MONTH(date) DESC";

    $result = $this->conn->query($query);

    if (!$result) {
        die("Query failed: " . $this->conn->error);
    }

    return $result->fetch_all(MYSQLI_ASSOC);
}
public function getClientDetailsByInvoiceId($invoice_id) {
    $query = "SELECT c.id, c.Institution, c.ClientName, c.PhoneNumber, 
                     c.address, c.username, c.serviceCondition, 
                     c.created_at, c.last_update 
              FROM client c
              JOIN $this->table_invoices i ON c.id = i.client_id
              WHERE i.id = ?";

    $stmt = $this->conn->prepare($query);
    $stmt->bind_param("i", $invoice_id);
    $stmt->execute();
    $result = $stmt->get_result();
    
    return $result->fetch_assoc();
}

public function getInvoicesByClientId($client_id) {
    $query = "SELECT * FROM $this->table_invoices WHERE client_id = ? ORDER BY date DESC";
    $stmt = $this->conn->prepare($query);
    $stmt->bind_param("i", $client_id);
    $stmt->execute();
    $result = $stmt->get_result();
    return $result->fetch_all(MYSQLI_ASSOC);
}




public function getFilteredInvoices($month, $page, $perPage) {
    $offset = ($page - 1) * $perPage;
    $status = $_GET['status'] ?? '';
    
    $query = "SELECT * FROM {$this->table_invoices}
             WHERE DATE_FORMAT(`date`, '%Y-%m') = ?";
    
    if ($status) {
        $query .= " AND `status` = ?";
    }
    
    $query .= " ORDER BY `date` DESC LIMIT ? OFFSET ?";
    
    $stmt = $this->conn->prepare($query);
    if ($stmt === false) {
        error_log("Failed to prepare query: " . $this->conn->error);
        return [];
    }
    
    if ($status) {
        $stmt->bind_param("ssii", $month, $status, $perPage, $offset);
    } else {
        $stmt->bind_param("sii", $month, $perPage, $offset);
    }
    
    if (!$stmt->execute()) {
        error_log("Query execution failed: " . $stmt->error);
        return [];
    }
    
    $result = $stmt->get_result();
    return $result->fetch_all(MYSQLI_ASSOC);
}



public function getTotalFilteredInvoices($month) {
    $status = $_GET['status'] ?? '';
    
    $query = "SELECT COUNT(*) as total FROM {$this->table_invoices}
             WHERE DATE_FORMAT(`date`, '%Y-%m') = ?";
    
    if ($status) {
        $query .= " AND `status` = ?";
    }
    
    $stmt = $this->conn->prepare($query);
    if ($stmt === false) {
        error_log("Failed to prepare query: " . $this->conn->error);
        return 0;
    }
    
    if ($status) {
        $stmt->bind_param("ss", $month, $status);
    } else {
        $stmt->bind_param("s", $month);
    }
    
    if (!$stmt->execute()) {
        error_log("Query execution failed: " . $stmt->error);
        return 0;
    }
    
    $result = $stmt->get_result();
    $row = $result->fetch_assoc();
    return $row['total'] ?? 0;
}



public function getInvoiceStatistics($month) {
    $status = $_GET['status'] ?? '';
    
    $query = "SELECT 
                COUNT(id) as total_invoices,
                COALESCE(SUM(total_amount), 0) as total_amount,
                COALESCE(SUM(paid_amount), 0) as total_paid,
                COALESCE(SUM(due_amount), 0) as total_due
             FROM {$this->table_invoices}
              WHERE DATE_FORMAT(`date`, '%Y-%m') = ?";
    
    if ($status) {
        $query .= " AND `status` = ?";
    }
    
    $stmt = $this->conn->prepare($query);
    if ($stmt === false) {
        error_log("Failed to prepare query: " . $this->conn->error);
        return [
            'total_invoices' => 0,
            'total_amount' => 0,
            'total_paid' => 0,
            'total_due' => 0
        ];
    }
    
    if ($status) {
        $stmt->bind_param("ss", $month, $status);
    } else {
        $stmt->bind_param("s", $month);
    }
    
    if (!$stmt->execute()) {
        error_log("Query execution failed: " . $stmt->error);
        return [
            'total_invoices' => 0,
            'total_amount' => 0,
            'total_paid' => 0,
            'total_due' => 0
        ];
    }
    
    $result = $stmt->get_result();
    return $result->fetch_assoc();
}
    
    
    
    
    
    
    
    
    
public function createInvoiceWithValidation($client_id, $total_amount, $items = []) {
    // Validate input
    if (!is_numeric($client_id) || $client_id <= 0) {
        return [
            'success' => false,
            'message' => 'Invalid client ID',
            'invoice_number' => null
        ];
    }

    if (!is_numeric($total_amount) || $total_amount <= 0) {
        return [
            'success' => false,
            'message' => 'Invalid total amount',
            'invoice_number' => null
        ];
    }

    // Check for potential duplicate invoice
    list($isDuplicate, $duplicateInvoiceNumber) = $this->isPotentialDuplicate($client_id, $total_amount, $items);
    if ($isDuplicate) {
        return [
            'success' => false,
            'message' => 'Potential duplicate invoice detected',
            'invoice_number' => $duplicateInvoiceNumber
        ];
    }

    // Start transaction
    $this->conn->begin_transaction();

    try {
        // Create the invoice
        $invoice_id = $this->createInvoice($client_id, $total_amount);
        
        if (!$invoice_id) {
            throw new Exception("Failed to create invoice");
        }

        // Add items to the invoice
        foreach ($items as $item) {
            $added = $this->addInvoiceItem(
                $invoice_id,
                $item['product_name'],
                $item['quantity'],
                $item['unit_price'],
                $item['description']
            );
            
            if (!$added) {
                throw new Exception("Failed to add invoice items");
            }
        }

        // Get the complete invoice with number
        $invoice = $this->getInvoiceDetails($invoice_id);
        
        if (!$invoice) {
            throw new Exception("Failed to retrieve created invoice");
        }

        // Commit transaction
        $this->conn->commit();

        return [
            'success' => true,
            'invoice_number' => $invoice['invoice_number'],
            'message' => 'Invoice created successfully'
        ];

    } catch (Exception $e) {
        $this->conn->rollback();
        error_log("Invoice creation failed: " . $e->getMessage());
        
        return [
            'success' => false,
            'message' => 'Invoice creation failed: ' . $e->getMessage(),
            'invoice_number' => null
        ];
    }
}

/**
 * Check for potential duplicate invoice and return invoice number if found
 * @param int $client_id
 * @param float $total_amount
 * @param array $items
 * @return array [bool $isDuplicate, string|null $invoice_number]
 */
private function isPotentialDuplicate($client_id, $total_amount, $items) {
    $query = "SELECT invoice_number 
             FROM {$this->table_invoices} 
             WHERE client_id = ? 
             AND total_amount = ?
             AND date >= DATE_SUB(NOW(), INTERVAL 1 DAY)
             LIMIT 1";
    
    $stmt = $this->conn->prepare($query);
    $stmt->bind_param("id", $client_id, $total_amount);
    $stmt->execute();
    $result = $stmt->get_result()->fetch_assoc();

    if ($result) {
        return [true, $result['invoice_number']];
    }

    // If items are provided, check for similar items in recent invoices
    if (!empty($items)) {
        $productNames = array_column($items, 'product_name');
        $placeholders = implode(',', array_fill(0, count($productNames), '?'));
        
        $query = "SELECT i.invoice_number
                 FROM {$this->table_items} bi
                 JOIN {$this->table_invoices} i ON bi.invoice_id = i.id
                 WHERE i.client_id = ?
                 AND bi.product_name IN ($placeholders)
                 AND i.date >= DATE_SUB(NOW(), INTERVAL 3 DAY)
                 LIMIT 1";
        
        $stmt = $this->conn->prepare($query);
        $types = str_repeat('s', count($productNames));
        $stmt->bind_param("i$types", $client_id, ...$productNames);
        $stmt->execute();
        $result = $stmt->get_result()->fetch_assoc();

        if ($result) {
            return [true, $result['invoice_number']];
        }
    }

    return [false, null];
}
    
    
    
    
}
?>
