<?php
class UserSubscriptionModel {
    private $conn;
    private $table_rates = 'rate';
    private $table_licenses = 'license';

    public function __construct() {
        $database = new Database();
        $this->conn = $database->getConnection();
    }

    // ==================== RATE OPERATIONS ====================

    // Create new rate entry
    public function createRate($user_id, $student_id, $rate) {
        $query = "INSERT INTO {$this->table_rates} (user, student, rate) VALUES (?, ?, ?)";
        $stmt = $this->conn->prepare($query);
        $stmt->bind_param("sid", $user_id, $student_id, $rate);
        return $stmt->execute();
    }

    // Get all rates
    public function getAllRates() {
        $query = "SELECT id, user, student, rate FROM {$this->table_rates}";
        $result = $this->conn->query($query);
        return $result->fetch_all(MYSQLI_ASSOC);
    }

    // Get rate by ID
    public function getRateById($id) {
        $query = "SELECT id, user, student, rate FROM {$this->table_rates} WHERE id = ?";
        $stmt = $this->conn->prepare($query);
        $stmt->bind_param("i", $id);
        $stmt->execute();
        return $stmt->get_result()->fetch_assoc();
    }

    // Get rates by user
    public function getRatesByUser($user_id) {
        $query = "SELECT id, user, student, rate FROM {$this->table_rates} WHERE user = ?";
        $stmt = $this->conn->prepare($query);
        $stmt->bind_param("i", $user_id);
        $stmt->execute();
        return $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
    }

    // Update rate
    public function updateRate($id, $student_id, $rate) {
        $query = "UPDATE {$this->table_rates} SET student = ?, rate = ? WHERE id = ?";
        $stmt = $this->conn->prepare($query);
        $stmt->bind_param("idi", $student_id, $rate, $id);
        return $stmt->execute();
    }

    // Delete rate
    public function deleteRate($id) {
        $query = "DELETE FROM {$this->table_rates} WHERE id = ?";
        $stmt = $this->conn->prepare($query);
        $stmt->bind_param("i", $id);
        return $stmt->execute();
    }

    // ==================== LICENSE OPERATIONS ====================

    // Create new license entry
    public function createLicense($user_id, $date) {
        $query = "INSERT INTO {$this->table_licenses} (user, date) VALUES (?, ?)";
        $stmt = $this->conn->prepare($query);
        $stmt->bind_param("is", $user_id, $date);
        return $stmt->execute();
    }

    // Get all licenses
    public function getAllLicenses() {
        $query = "SELECT id, user, date FROM {$this->table_licenses}";
        $result = $this->conn->query($query);
        return $result->fetch_all(MYSQLI_ASSOC);
    }

    // Get license by ID
    public function getLicenseById($id) {
        $query = "SELECT id, user, date FROM {$this->table_licenses} WHERE id = ?";
        $stmt = $this->conn->prepare($query);
        $stmt->bind_param("i", $id);
        $stmt->execute();
        return $stmt->get_result()->fetch_assoc();
    }

    // Get license by user
    public function getLicenseByUser($user_id) {
        $query = "SELECT id, user, date FROM {$this->table_licenses} WHERE user = ?";
        $stmt = $this->conn->prepare($query);
        $stmt->bind_param("i", $user_id);
        $stmt->execute();
        return $stmt->get_result()->fetch_assoc();
    }

    // Update license date (specific to your requirement)
    public function updateLicenseDate($user_id, $new_date) {
        $query = "UPDATE {$this->table_licenses} SET date = ? WHERE id = ?";
        $stmt = $this->conn->prepare($query);
        $stmt->bind_param("si", $new_date, $user_id);
        
        return $stmt->execute();
    }

    // Delete license
    public function deleteLicense($id) {
        $query = "DELETE FROM {$this->table_licenses} WHERE id = ?";
        $stmt = $this->conn->prepare($query);
        $stmt->bind_param("i", $id);
        return $stmt->execute();
    }

    // ==================== COMBINED OPERATIONS ====================

    // Get complete user subscription data (rates + license)
    public function getUserSubscriptionData($user_id) {
        return [
            'rates' => $this->getRatesByUser($user_id),
            'license' => $this->getLicenseByUser($user_id)
        ];
    }

    // Check if user has active license (based on date)
    public function hasActiveLicense($user_id) {
        $license = $this->getLicenseByUser($user_id);
        if (!$license) return false;
        
        $current_date = date('Y-m-d');
        return ($license['date'] >= $current_date);
    }
}